#include <string>
#include <map>
#include <set>
#include "types.h"
#include "lib\curses.h"

using namespace std;

extern "C" {
extern void configure_game();
extern byte new_game();
extern void system_init();

extern signed int white_weights[GAME_PARTS][e_max_weight];
extern signed int black_weights[GAME_PARTS][e_max_weight];
};


class CPlayer {
	unsigned int wins;
public:
	signed int weights[GAME_PARTS][e_max_weight];
	CPlayer()
	{
		wins=0;
		memset(weights,0,sizeof(weights));
	}
	bool operator<(const CPlayer &r) const
	{
		if (wins<r.wins)
			return true;
		return false;
	}
	void CrossWith(const CPlayer &r)
	{
		size_t p,w;
		for (p=0;p<GAME_PARTS;++p)
		{
			for (w=0;w<e_max_weight;++w)
			{
				if (rand()%4==0)
					weights[p][w]=r.weights[p][w];
			}
		}
	}
	void Mutate()
	{
		size_t p,w;
		for (p=0;p<GAME_PARTS;++p)
		{
			for (w=0;w<e_max_weight;++w)
			{
				while (rand()%2==0)
					weights[p][w]+=rand()%7-3;
			}
		}
	}
	void IncWins()
	{
		++wins;
	}
	unsigned int GetWins()
	{
		return wins;
	}
	void ZeroWins()
	{
		wins=0;
	}
};

class CTournament {
	unsigned long games_played;
public:
	CPlayer players[MAX_PLAYERS];
	CPlayer best_players[BEST_PLAYERS];

	void Init()
	{
		size_t p;

		games_played=0;
		LoadBestPlayers();
		for (p=0;p<MAX_PLAYERS;++p)
		{
			if (players[p].GetWins()==0)
			{
				players[p].Mutate();
				players[p].Mutate();
				players[p].Mutate();
			}
		}
		configure_game();
	}

	void ChooseBestPlayers()
	{
		size_t i;
		multimap < unsigned int, CPlayer, greater<unsigned int> > best;
		multimap < unsigned int, CPlayer, greater<unsigned int> >::iterator m;
		for (i=0;i<MAX_PLAYERS;++i)
		{
			best.insert( pair <unsigned int,CPlayer> (players[i].GetWins(),players[i]) );
		}
		for (i=0,m=best.begin();i<BEST_PLAYERS;++i,++m)
		{
			best_players[i]=m->second;
		}
	}

	void ReproduceBestPlayers()
	{
		size_t i,j,k;
		
		for (i=0;i<BEST_PLAYERS;++i)
		{
			players[i]=best_players[i];
			for (k=0;k<BEST_PLAYERS-1;++k)
			{
				j=BEST_PLAYERS+i*(BEST_PLAYERS-1)+k;
				players[j]=best_players[i];
			}
		}
	}

	void CrossWithBestPlayers()
	{
		size_t i,j,k,l;
		for (i=0;i<BEST_PLAYERS;++i)
		{
			l=0;
			for (k=0;k<BEST_PLAYERS;++k)
			{
				if (k!=i)
				{
					j=BEST_PLAYERS+i*(BEST_PLAYERS-1)+l;
					players[j].CrossWith(players[k]);
					players[j].Mutate();
					++l;
				}
			}
		}

		// set my own
		// part 0
		players[BEST_PLAYERS].weights[0][e_piece_differential_weight]=0;
		players[BEST_PLAYERS].weights[0][e_potential_mobility_weight]=7;
		players[BEST_PLAYERS].weights[0][e_mobility_weight]=7;
		players[BEST_PLAYERS].weights[0][e_edges_weight]=0;
		players[BEST_PLAYERS].weights[0][e_corners_weight]=35;
		players[BEST_PLAYERS].weights[0][e_x_weight]=-8;
		players[BEST_PLAYERS].weights[0][e_c_weight]=-4;
		players[BEST_PLAYERS].weights[0][e_stable_disc_weight]=8;

		// part 1
		players[BEST_PLAYERS].weights[1][e_piece_differential_weight]=0;
		players[BEST_PLAYERS].weights[1][e_potential_mobility_weight]=7;
		players[BEST_PLAYERS].weights[1][e_mobility_weight]=6;
		players[BEST_PLAYERS].weights[1][e_edges_weight]=3;
		players[BEST_PLAYERS].weights[1][e_corners_weight]=35;
		players[BEST_PLAYERS].weights[1][e_x_weight]=-8;
		players[BEST_PLAYERS].weights[1][e_c_weight]=-4;
		players[BEST_PLAYERS].weights[1][e_stable_disc_weight]=8;

		// part 2
		players[BEST_PLAYERS].weights[2][e_piece_differential_weight]=0;
		players[BEST_PLAYERS].weights[2][e_potential_mobility_weight]=7;
		players[BEST_PLAYERS].weights[2][e_mobility_weight]=5;
		players[BEST_PLAYERS].weights[2][e_edges_weight]=4;
		players[BEST_PLAYERS].weights[2][e_corners_weight]=35;
		players[BEST_PLAYERS].weights[2][e_x_weight]=-8;
		players[BEST_PLAYERS].weights[2][e_c_weight]=-4;
		players[BEST_PLAYERS].weights[2][e_stable_disc_weight]=4;

		// part 3
		players[BEST_PLAYERS].weights[3][e_piece_differential_weight]=4;
		players[BEST_PLAYERS].weights[3][e_potential_mobility_weight]=1;
		players[BEST_PLAYERS].weights[3][e_mobility_weight]=4;
		players[BEST_PLAYERS].weights[3][e_edges_weight]=5;
		players[BEST_PLAYERS].weights[3][e_corners_weight]=35;
		players[BEST_PLAYERS].weights[3][e_x_weight]=-8;
		players[BEST_PLAYERS].weights[3][e_c_weight]=-4;
		players[BEST_PLAYERS].weights[3][e_stable_disc_weight]=4;

		// part 4
		players[BEST_PLAYERS].weights[4][e_piece_differential_weight]=1;
		players[BEST_PLAYERS].weights[4][e_potential_mobility_weight]=0;
		players[BEST_PLAYERS].weights[4][e_mobility_weight]=0;
		players[BEST_PLAYERS].weights[4][e_edges_weight]=0;
		players[BEST_PLAYERS].weights[4][e_corners_weight]=0;
		players[BEST_PLAYERS].weights[4][e_x_weight]=0;
		players[BEST_PLAYERS].weights[4][e_c_weight]=0;
		players[BEST_PLAYERS].weights[4][e_stable_disc_weight]=0;
	}

	void Main()
	{
		size_t i;
		Init();
		for (i=0;i<1000;++i)
		{
			mvprintw(20,0,"Round %d",i);
			refresh();

			PlayALLvsALL();
			ChooseBestPlayers();
			SaveBestPlayers();
			ReproduceBestPlayers();
			CrossWithBestPlayers();
		}
	}
	void PlayALLvsALL()
	{
		size_t i,j;
		for (i=0;i<MAX_PLAYERS;++i)
			players[i].ZeroWins();

		for (i=0;i<MAX_PLAYERS;++i)
		{
			for (j=0;j<MAX_PLAYERS;++j)
			{
				if (i!=j)
				{
					PlayGame(i,j);
					PlayGame(j,i);
					PlayGame(i,j);
					PlayGame(j,i);
				}
			}
		}
	}
	void PlayGame(size_t black, size_t white)
	{
		memcpy(white_weights,players[white].weights,sizeof(white_weights));		
		memcpy(black_weights,players[black].weights,sizeof(black_weights));		
		byte winner=new_game();
		if (winner==WHITE)
			players[white].IncWins();
		else if (winner==BLACK)
			players[black].IncWins();

		++games_played;
		if (games_played%100==0)
			mvprintw(21,0,"Played %ld",games_played);
		refresh();
		return;
	}	
	
	void SaveBestPlayers()
	{
		FILE *fp=fopen("best.bin","wb+");
		if (fp!=NULL)
		{
			fwrite(best_players,sizeof(best_players),1,fp);
			fclose(fp);
		}
	}

	bool LoadBestPlayers()
	{
		FILE *fp=fopen("best.bin","rb");
		if (fp!=NULL)
		{
			fread(best_players,sizeof(best_players),1,fp);
			memcpy(players,best_players,sizeof(best_players));
			fclose(fp);

			ReproduceBestPlayers();
			CrossWithBestPlayers();
			return true;
		}
		return false;
	}
};

void main(void)
{
	system_init();

	CTournament tournament;
#if !GENETIC_MODE
	tournament.LoadBestPlayers();
	tournament.Init();
	memcpy(white_weights,tournament.players[0].weights,sizeof(white_weights));		
	memcpy(black_weights,tournament.players[0].weights,sizeof(black_weights));	
	new_game();
#else
	tournament.Main();
#endif
}